##
## Title: Mesolithic Hunter-Gatherer Adaptations to Small Mediterranean Island Ecosystems
##
## Authors: Mario Mata-González*, Mathew Stewart*, James A. Blinkhorn, Alexander F. Blackwood, 
##          Jop van Dijen, Carli A. E. Peters, Ethel Allué, Aitor Burguet-Coca, 
##          Andrés Currás, Huw S. Groucutt, Nicholas C. Vella, Eleanor M. L. Scerri
##
##
## Supplementary Data - Code
## Extract Supplementary_data.zip to a folder on the hard-drive, then set that folder (Supplementary Data) as the working directory


# Required packages:

library(sf)
library(terra)
library(raster)
library(spatstat)
library(tmap)
library(tidyverse)
library(ggpattern)
library(ggspatial)

setwd("...") # Select location of "Supplementary Data" folder to set working directory

# Note: - If pattern fill on geom_sf_pattern() doesnt work, try install latest version of ggpattern: remotes::install_github("coolbutuseless/ggpattern")

#===============================================================================
# 
# ------------------------------------------------------------------------------
# Load necessary datasets


# 1) Load Latnija Faunal DB for spatial analysis

lat_fauna_sq <- read.csv("data/lat.fauna.by.square.csv") # Load faunal DB - plotted finds and sieved material by grid square
lat_fauna <- read.csv("data/lat.fauna.spatialdb.csv", colClasses = c(Latitude = "character", Longitude = "character")) # Load faunal DB - plotted finds

lat_fauna$Latitude  <- as.numeric(lat_fauna$Latitude) # to avoid R truncating the coordinates...
lat_fauna$Longitude <- as.numeric(lat_fauna$Longitude)
lat_fauna$long = lat_fauna$Longitude # duplicate long and lat fields so they remain after conversion to shapefile
lat_fauna$lat = lat_fauna$Latitude
options(digits = 10) # display more decimals to check if coordinates were properly transfered
head(lat_fauna[, c("Latitude", "lat", "Longitude", "long")]) # check duplicated coordinates

# Check for duplicate points

lat_dup_uids <- lat_fauna |> # Check for duplicate coordinates within each 'type' of the lat_fauna dataset
  group_by(type, Longitude, Latitude) |>
  filter(n() > 1) |>
  summarise(
    n = n(),
    UIDs = paste(UID, collapse = ", "),
    .groups = "drop"
  ) |>
  arrange(type, desc(n))

if (nrow(lat_dup_uids) == 0) { # Check the table that was just created - it will list duplicates
  message("No duplicates detected (within type, by Longitude/Latitude).")
} else {
  lat_dup_uids
}

length(unique(lat_fauna$UID)) # Return total number of unique specimens to be used in the following analysis

lat_fauna %>%
  group_by(type) %>%
  summarise(n_unique_UID = n_distinct(UID)) # print a table showing how many unique specimens under each 'type'


## 3) Convert dataframe to shapefile and export

lat_fauna_sf <- st_as_sf(
  lat_fauna,
  coords = c("Longitude", "Latitude"),
  crs = 32633  # WGS84 UTM zone 33N
)

st_write(lat_fauna_sf, "shapefiles/lat_fauna.shp", append = FALSE)  # save as shapefile, overwriting any previous files with same name

## 4) Import points shapefile

lat <- sf::st_read("shapefiles/lat_fauna.shp") # Spatial and Faunal DB for KDEs

# ------------------------------------------------------------------------------
# STUDY AREA 
## 1) Import the Latnija excavation polygons

lat_area <- sf::st_read("shapefiles/LAT_StudyArea.shp")  # Study Area bounding box
lat_limestone <- sf::st_read("shapefiles/LAT_StudyArea_Extracted.shp")
lat_cave <- sf::st_read("shapefiles/LAT_Cave_Outline.shp")
lat_dripline <- sf::st_read("shapefiles/LAT_Cave_Dripline.shp")
lat_grid <- sf::st_read("shapefiles/LAT_ExcavationGrid_withoutT4.shp")
lat_t4grid_nophaseV <- sf::st_read("shapefiles/LAT_ExcavationGrid_T4_No_PhaseV.shp")
lat_t4grid_phaseV <- sf::st_read("shapefiles/LAT_ExcavationGrid_T4_PhaseV.shp")
lat_t4outline <- sf::st_read("shapefiles/LAT_T4_ExcavationOutline.shp")
lat_t4rocks <- sf::st_read("shapefiles/LAT_T4_Rocks.shp")
lat_t4grid <- sf::st_read("shapefiles/LAT_ExcavationGrid_T4.shp")

lat_phasev <- sf::st_read("shapefiles/LAT2024_PhaseV_utm33N.shp") # Import full Spatial DB for plotting Phase V points

## 2) Plot points and polygons together using tmap (to view shapefiles and check they are properly imported)

tm_shape(lat_grid) +
  tm_polygons() +
  tm_shape(lat_area) +
  tm_polygons(col = NA,alpha = 0.1, border.col = NA) +
  tm_shape(lat_t4grid_phaseV) +
  tm_polygons(col = NA,alpha = 0.1, border.col = "black", lwd = 0.5) +
  tm_shape(lat_t4grid_nophaseV) +
  tm_polygons(border.col = "black", lwd = 0.5) +
  tm_shape(lat_t4outline) +
  tm_polygons(col = NA,alpha = 0.1, border.col = "black", lwd = 2)+
  tm_shape(lat_dripline) +
  tm_polygons(col = NA,alpha = 0.1, border.col = "black", lwd = 2, lty = 2)+
  tm_shape(lat_phasev) +
  tm_dots()


# ------------------------------------------------------------------------------
# Analysis of marked patterns (Latnija Phase V Fauna 'types')
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# 1) Function to generate KDE rasters for all unique types (marks)
# Note: Sigma (banwidth) for the density estimate is calculated using Likelihood 
#       Cross Validation Bandwidth Selection from spatstat for each 'mark'
#       dimyx = 500 for size/resolution balance. Change as necessary
#       window = can be changed to any of the owin converted polygons
# ------------------------------------------------------------------------------

kde_raster_all_types <- function(sf_points, type_field, window, dimyx = 1000) {
  # Ensure sf_points has planar coordinates
  if (st_is_longlat(sf_points)) {
    sf_points <- st_transform(sf_points, 32633) # example UTM zone
  }
  
  types <- unique(sf_points[[type_field]])
  
  kde_list <- lapply(types, function(t) {
    pts_sub <- sf_points[sf_points[[type_field]] == t, ]
    pts_ppp <- ppp(
      x = st_coordinates(pts_sub)[,1],
      y = st_coordinates(pts_sub)[,2],
      window = window
    )
    
    dens <- density(pts_ppp, sigma = bw.ppl(pts_ppp), dimyx = dimyx)
    kde_r <- raster(dens)
    crs(kde_r) <- st_crs(sf_points)$wkt
    return(kde_r)
  })
  
  names(kde_list) <- types
  return(kde_list)
}

# ------------------------------------------------------------------------------
# 2) Function to generate quadratcount plots for all unique types (marks)
# ------------------------------------------------------------------------------

quadrat_counts_all_types <- function(sf_points, type_field, window, nx = 20, ny = 20) {
  
  # Ensure planar coordinates
  if (st_is_longlat(sf_points)) {
    sf_points <- st_transform(sf_points, 32633) # example UTM zone
  }
  
  types <- unique(sf_points[[type_field]])
  
  qcp_list <- lapply(types, function(t) {
    # Subset points for this type
    pts_sub <- sf_points[sf_points[[type_field]] == t, ]
    
    # Convert to ppp
    pts_ppp <- ppp(
      x = st_coordinates(pts_sub)[,1],
      y = st_coordinates(pts_sub)[,2],
      window = window
    )
    
    # Quadrat counts
    qcp <- quadratcount(pts_ppp, nx = nx, ny = ny)
    
    return(qcp)
  })
  
  names(qcp_list) <- types
  return(qcp_list)
}

# ------------------------------------------------------------------------------
# 3) Function to compute scanLRTS and run eval.im for all types
# ------------------------------------------------------------------------------

scanLRTS_all_types <- function(sf_points, type_field, window) {
  
  # Ensure planar coordinates
  if (st_is_longlat(sf_points)) {
    sf_points <- st_transform(sf_points, 32633) # example UTM zone
  }
  
  types <- unique(sf_points[[type_field]])
  
  lrt_list <- lapply(types, function(t) {
    pts_sub <- sf_points[sf_points[[type_field]] == t, ]
    
    # Skip empty types
    if (nrow(pts_sub) == 0) return(NULL)
    
    pts_ppp <- ppp(
      x = st_coordinates(pts_sub)[,1],
      y = st_coordinates(pts_sub)[,2],
      window = window
    )
    
    # Run scanLRTS
    hs <- scanLRTS(
      pts_ppp,
      r = bw.ppl(pts_ppp),
      method = "poisson",
      alternative = "greater"
    )
    
    # Evaluate p-values
    pvals <- eval.im(pchisq(hs, df=1, lower.tail = FALSE))
    
    return(pvals)
  })
  
  names(lrt_list) <- types
  return(lrt_list)
}


# ------------------------------------------------------------------------------
# KDE steps
# ------------------------------------------------------------------------------

## 1) define windows for spatstat

lat_t4outline <- st_zm(lat_t4outline) # remove z dimension from excavation outline polygon

bb <- st_bbox(lat_t4outline) # expand bounding box by 1m on all sides (so none of the KDE rasters are cut off by edges of the excavation outline)
bb_expanded <- bb
bb_expanded["xmin"] <- bb["xmin"] - 1
bb_expanded["xmax"] <- bb["xmax"] + 1
bb_expanded["ymin"] <- bb["ymin"] - 1
bb_expanded["ymax"] <- bb["ymax"] + 1

lat_window <- as.owin(list(xrange = c(bb_expanded["xmin"], bb_expanded["xmax"]),
                           yrange = c(bb_expanded["ymin"], bb_expanded["ymax"]))) # define window with expanded bounding box. This window is useful for plotting maps in ggplot below.

lat_grid_owin <- as.owin(lat_t4grid_phaseV) # T4 phase V excavation squares as spatstat window
lat_outline_owin <- as.owin(lat_t4outline) # T4 outline as spatstat window


## 2) run custom functions for all 'marks' within lat_fauna --------------------

# Kernel Density
kde_all <- kde_raster_all_types(
  sf_points = lat,
  type_field = "type",
  window = lat_window
)

# Quadratcounts
qcp_list <- quadrat_counts_all_types(lat, type_field = "type", window = lat_outline_owin, nx = 20, ny = 20)

# ScanLRTS
lrt_results <- scanLRTS_all_types(lat, type_field = "type", window = lat_window)


## 3) Plot the results and then save -------------------------------------------


# 3.1) KDE plots without quadratcount plots (Saving to folder called KDE_plots)
for (type_name in names(kde_all)) {
  
  # Create output directory (safe to repeat)
  out_dir <- "outputs/KDE_plots"
  if (!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)
  
  # Output filename
  out_file <- file.path(out_dir, paste0("KDE_", type_name, ".png"))
  
  # Open graphics device
  png(filename = out_file, width = 6, height = 6, units = "in", res = 300)
  
  # ---- Your original plotting code ----
  r <- kde_all[[type_name]]
  e <- extent(r)
  
  plot(
    r,
    main = paste("KDE:", type_name),
    xlim = c(e@xmin, e@xmax),
    ylim = c(e@ymin, e@ymax),
    asp  = 1,
    axes = TRUE,
    box  = TRUE
  )
  # Close device and write file
  dev.off()
  cat("Rendering output for:", type_name, "to: ", normalizePath(out_dir), "\n")
}


# 3.2) KDE plots With quadratcounts (Saving to folder called KDE_QuadCount_plots)
for (type_name in names(kde_all)) {
  
  # Create output directory
  out_dir <- "outputs/KDE_QuadCount_Plots"
  if (!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)
  
  # Output filename
  out_file <- file.path(out_dir, paste0("KDE_QuadCounts_", type_name, ".png"))
  
  # Open graphics device
  png(filename = out_file, width = 6, height = 6, units = "in", res = 300)
  
  # ---- Your original plotting code ----
  
  # Extract raster and extent
  r <- kde_all[[type_name]]
  e <- extent(r)
  
  # Plot the KDE raster
  plot(
    r,
    main = paste("KDE + Quadrat:", type_name),
    xlim = c(e@xmin, e@xmax),
    ylim = c(e@ymin, e@ymax),
    asp  = 1,
    axes = TRUE,
    box  = TRUE
  )
  
  # Overlay quadrat counts
  if (!is.null(qcp_list[[type_name]])) {
    plot(
      qcp_list[[type_name]],
      add = TRUE,
      col = "red",
      lwd = 0.2,
      cex = 0.6
    )
  }
  # Close device and write file
  dev.off()
  cat("Rendering output for:", type_name, "to: ", normalizePath(out_dir), "\n")
}


# 3.3) Loop through and plot results of ScanLRT with p-values < 0.001 (Saving to folder called LRTS_plots)
for (type_name in names(lrt_results)) {
  
  # Create output directory (safe even if it already exists)
  out_dir <- "outputs/LRTS_plots"
  if (!dir.exists(out_dir)) dir.create(out_dir)
  
  pvals <- lrt_results[[type_name]]
  if (is.null(pvals)) next
  
  out_file <- file.path(out_dir, paste0("LRTS_", type_name, ".png"))
  png(filename = out_file, width = 2000, height = 2000, res = 300)
  
  plot(pvals < 0.001,
       main = paste("L.R.T.S. (p < 0.001):", type_name),
       cex.main = 1)
  
  plot(lat_window, add = TRUE)
  plot(lat_grid_owin, add = TRUE)
  plot(lat_outline_owin, add = TRUE)
  plot(st_geometry(lat_t4grid_nophaseV),
       col = NA, border = "black", add = TRUE)
  
  dev.off()
  cat("Rendering output for:", type_name, "to: ", normalizePath(out_dir), "\n")
}


# 3.4) Save all KDE rasters (without plot elements) to folder KDE_Rasters

for (type_name in names(kde_all)) {
  kde_dir <- "outputs/KDE_Rasters"
  if (!dir.exists(kde_dir)) dir.create(kde_dir, recursive = TRUE)
  filename <- file.path(kde_dir, paste0("KDE_", type_name, ".tif"))
  writeRaster(kde_all[[type_name]], filename, overwrite = TRUE)
  cat("Rendering output for:", type_name, "to: ", normalizePath(out_dir), "\n")
}


#-------------------------------------------------------------------------------
# Plot results in R to make figures
#-------------------------------------------------------------------------------

## 1) Function to plot and save KDE rasters with excavation polygons

plot_kde_type <- function(
    kde_all,
    points_sf,
    type_name,
    lat_grid,
    lat_t4grid_nophaseV,
    lat_t4grid_phaseV,
    lat_outline,
    lat_window,
    lat_cave,
    lat_limestone,
    lat_t4rocks,
    lat_dripline,
    pattern_file = "textures/limestone.png"
) {
  
  # --- 1. Subset points by type ---
  pts_type <- points_sf[points_sf$type == type_name, ]
  
  # --- 2. Convert window and outline to sf polygons ---
  window_poly <- as.polygonal(lat_window)
  window_sf <- sf::st_as_sf(window_poly)
  sf::st_crs(window_sf) <- sf::st_crs(points_sf)
  
  outline_sf <- sf::st_as_sf(lat_outline)
  sf::st_crs(outline_sf) <- sf::st_crs(points_sf)
  
  # --- 3. Convert KDE raster and crop to window ---
  kde_rast <- terra::rast(kde_all[[type_name]])
  #kde_rast <- terra::project(kde_rast, sf::st_crs(window_sf)$proj4string) # reproject with smoothing
  kde_rast <- terra::project(kde_rast,sf::st_crs(window_sf)$proj4string,method = "near") # Reproject without smoothing
  
  kde_crop <- terra::crop(kde_rast,terra::vect(outline_sf))
  kde_crop <- terra::mask(kde_crop,terra::vect(outline_sf))
  
  # Downscale raster to prevent cowplot crashing
  #kde_rast_small <- terra::aggregate(kde_crop, fact = 2) # will downscale KDE raster, if needed
  kde_rast_small <- kde_crop # remove this if downscaling is used
  kde_df <- as.data.frame(kde_rast_small, xy = TRUE)
  names(kde_df) <- c("x", "y", "value")
  
  #kde_df <- kde_df[kde_df$value > -1, ]   # remove low values
  
  # --- 4. Crop vector layers to window ---
  lat_grid_crop <- sf::st_intersection(lat_grid, window_sf)
  lat_t4grid_nophaseV_crop <- sf::st_intersection(lat_t4grid_nophaseV, window_sf)
  lat_t4grid_phaseV_crop <- sf::st_intersection(lat_t4grid_phaseV, window_sf)
  lat_outline_crop <- sf::st_intersection(lat_outline, window_sf)
  lat_cave_crop <- sf::st_intersection(lat_cave, window_sf)
  lat_dripline_crop <- sf::st_intersection(lat_dripline, window_sf)
  lat_limestone_crop <- sf::st_intersection(lat_limestone, window_sf)
  
  # --- 5. Prepare square labels ---
  label_squares <- lat_t4grid_nophaseV_crop %>%
    filter(SQ_ID %in% c("J2","J3","J4","J5","J6","K6","L6","M6","N6")) %>%
    st_centroid() %>%
    mutate(
      label_text = case_when(
        SQ_ID %in% c("J2","J3","J4","J5") ~ substr(SQ_ID, 2, 2),
        SQ_ID == "J6" ~ "J",
        SQ_ID %in% c("K6","L6","M6","N6") ~ substr(SQ_ID, 1, 1)
      ),
      nudge = case_when(
        SQ_ID %in% c("J2","J3","J4","J5") ~ -0.5,
        SQ_ID == "J6" ~ 0.5,
        SQ_ID %in% c("K6","L6","M6","N6") ~ 0.5
      )
    ) %>%
    bind_rows(
      lat_t4grid_nophaseV_crop %>%
        filter(SQ_ID == "J6") %>%
        st_centroid() %>%
        mutate(
          label_text = "6",
          nudge = -0.5
        )
    )
  # --- 6. Define the title - Format type_name for the title  ---
  plot_name <- dplyr::recode(
    type_name,
    "mammals" = "Mammals",
    "aves" = "Aves",
    "burnt" = "Burnt",
    "unburnt" = "Unburnt",
    "nocrust" = "No Crusts",
    "crust" = "Crusts",
    .default = type_name  # fallback
  )
  
  plot_title <- paste("Latnija Phase V Fauna:", plot_name)
  
  # --- 7. Build the map ---
  map <- ggplot() +
    # Patterned limestone
    ggpattern::geom_sf_pattern(
      data = lat_limestone_crop,
      aes(geometry = geometry),
      inherit.aes = FALSE,
      fill = NA,
      color = "black",
      pattern = "image",
      pattern_filename = pattern_file,
      pattern_type = "tile",
      pattern_scale = 1.5,
      pattern_aspect_ratio = 1
    ) +
    
    # Points (all of Phase V) - Add if required
    # geom_sf(data = lat_phasev, color = "black", size = 0.05, alpha = 1,show.legend = FALSE) +
    
    # Continuous KDE raster using viridis
    geom_raster(
      data = kde_df,
      aes(x = x, y = y, fill = value),
      alpha = 1,
      interpolate = FALSE
    ) +
    scale_fill_viridis_c(
      option = "viridis",
      name = paste("Density:"),
      guide = guide_colorbar(order = 2)
    ) +
    
    # Grids and outlines
    geom_sf(data = lat_grid_crop, fill = "grey", alpha = 0.6, color = "grey60", linewidth = 0.3) +
    geom_sf(data = lat_cave_crop, fill = NA, color = "black", linewidth = 0.8) +
    geom_sf(data = lat_t4grid_phaseV_crop, fill = NA, color = "grey60", linewidth = 0.3) +
    geom_sf(data = lat_t4grid_nophaseV_crop, fill = "lightgrey", alpha = 0.3, color = NA, linewidth = 0.3) +
    # Patterned rocks with legend
    ggpattern::geom_sf_pattern(
      data = lat_t4rocks,
      aes(geometry = geometry, pattern = "Rock"),
      inherit.aes = FALSE,
      fill = NA,
      color = "black",
      linewidth = 0.5,
      pattern_density = 0.03,
      pattern_spacing = 0.02,
      show.legend = TRUE
    ) +
    scale_pattern_manual(
      name = "",
      values = c("Rock" = "stripe"),
      guide = guide_legend(order = 1)
    ) +
    geom_sf(data = lat_t4grid_nophaseV_crop, fill = NA, alpha = 0.3, color = "grey60", linewidth = 0.3) +
    geom_sf(data = lat_dripline_crop, fill = NA, color = "black", linewidth = 0.6, linetype = "dashed") +
    geom_sf(data = lat_outline_crop, fill = NA, color = "black", linewidth = 0.6) +
    # Square labels
    geom_sf_text(
      data = label_squares,
      aes(label = label_text),
      nudge_x = label_squares$nudge,
      nudge_y = -0.6,
      size = 4,
      fontface = "bold",
      color = "black"
    ) +
    # North arrow
    annotation_north_arrow(
      location = "bl",
      which_north = "true",
      style = north_arrow_orienteering,
      pad_y = unit(0.9, "cm"),
      width = unit(0.5, "cm"),
      height = unit(1, "cm")
    ) +
    annotation_scale(
      location = "bl",
      width_hint = 0.2,
      text_cex = 1,
      text_face = "bold",
    ) +
    coord_sf(
      datum = sf::st_crs(lat_outline),
      expand = FALSE,
      clip = "on"
    ) +
    ggtitle(plot_title) +
    theme_minimal() +
    theme(
      plot.title = element_text(face = "bold", hjust = 0.5, size = 14),
      panel.grid = element_blank(),
      axis.ticks.length = unit(0.1, "cm"),
      axis.ticks = element_line(color = "black", linewidth = 0.5),
      axis.text = element_text(size = 8, color = "black"),
      axis.title = element_blank(),
      legend.position = "right",
      panel.border = element_rect(colour = "black", fill=NA, linewidth=1)
    )
  
  return(map)
}

## 2) View all maps in R and save to outputs/KDE_Maps


for (type_name in names(kde_all)) {
  # Create output directory
  out_dir <- "outputs/KDE_Maps"
  if (!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)
  
  p <- plot_kde_type(
    kde_all = kde_all,
    points_sf = lat,
    type_name = type_name,
    lat_grid = lat_grid,
    lat_t4grid_nophaseV = lat_t4grid_nophaseV,
    lat_t4grid_phaseV = lat_t4grid_phaseV,
    lat_outline = lat_t4outline,
    lat_window = lat_window,
    lat_cave = lat_cave,
    lat_limestone = lat_limestone,
    lat_t4rocks = lat_t4rocks,
    lat_dripline = lat_dripline
  )
  
  # Plot
  print(p)
  
  # Save to file
  ggsave(
    filename = file.path(out_dir, paste0("KDE_Map_", type_name, ".png")),
    plot = p,
    width = 6,
    height = 6,
    dpi = 600
  )
  cat("Rendering output for:", type_name, "to: ", normalizePath(out_dir), "\n")
}


# 3) Plot a single map in R (if needed)
plot_kde_type(
  kde_all = kde_all,
  points_sf = lat,
  type_name = 'nocrust', # Change this to desired 'mark' (dont forget the '')
  lat_grid = lat_grid,
  lat_t4grid_nophaseV = lat_t4grid_nophaseV,
  lat_t4grid_phaseV = lat_t4grid_phaseV,
  lat_outline = lat_t4outline,
  lat_window = lat_window,
  lat_cave = lat_cave,
  lat_limestone = lat_limestone,
  lat_t4rocks = lat_t4rocks,
  lat_dripline = lat_dripline
)

### ----------------------------------------------------------------------------
## Plot faunal data as profiles using (XZ) axes
### ----------------------------------------------------------------------------

# Get unique types
types <- unique(lat$type)

# Initialize a list to store plots
profile_plots <- list()

# Loop through each type and save to folder called Excavation_profiles

for (t in types) {
  out_dir <- "outputs/Excavation_profiles"
  if (!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)
  p <- lat %>%
    filter(type == t) %>%
    ggplot(aes(x = long, y = Altitud, color = type)) +
    geom_point(size = 0.5, alpha = 0.7, show.legend = FALSE) +
    scale_color_viridis_d(option = "D") +
    scale_y_continuous(limits = c(46, 48)) +
    scale_x_continuous(limits = c(439516, 439522), expand = expansion(mult = c(0, 0.1))) +
    labs(
      x = "Easting (UTM zone 33N)",
      y = "Elevation (m.asl)",
      color = "Faunal Type",
      title = paste("Profile:", t)
    ) +
    coord_sf(datum = sf::st_crs(lat), expand = FALSE) +
    scale_fill_viridis_c(option = "D", na.value = "white") +
    theme_minimal() +
    theme(
      plot.title = element_text(face = "bold", hjust = 0.5, size = 14),
      axis.ticks.length = unit(0.1, "cm"),
      axis.ticks = element_line(color = "black", linewidth = 0.5),
      axis.text = element_text(size = 8, color = "black"),
      panel.border = element_rect(color = "black", fill = NA, linewidth = 0.5),
      plot.margin = margin(t = 5, r = 20, b = 5, l = 5)
    )
  
  # Store
  profile_plots[[t]] <- p
  
  # Print immediately
  print(p)
  
  # Optional: Save at the same time
  ggsave(
    filename = file.path(out_dir, paste0("Profile_", t, ".png")),
    plot = p,
    width = 6,
    height = 3,
    dpi = 600,
    units = "in"
  )
  cat("Rendering output for:", t, "to: ", normalizePath(out_dir), "\n")
}

### ----------------------------------------------------------------------------
## Plot faunal data as planview maps
### ----------------------------------------------------------------------------

plot_planview_type <- function(
    points_sf,
    type_name,
    lat_grid,
    lat_t4grid_nophaseV,
    lat_t4grid_phaseV,
    lat_outline,
    lat_window,
    lat_cave,
    lat_limestone,
    lat_t4rocks,
    lat_dripline,
    pattern_file = "textures/limestone.png"
) {
  
  # --- 1. Subset points by type ---
  pts_type <- points_sf[points_sf$type == type_name, ]
  
  # --- 2. Convert window and outline to sf polygons ---
  window_poly <- as.polygonal(lat_window)
  window_sf <- sf::st_as_sf(window_poly)
  sf::st_crs(window_sf) <- sf::st_crs(points_sf)
  
  outline_sf <- sf::st_as_sf(lat_outline)
  sf::st_crs(outline_sf) <- sf::st_crs(points_sf)
  
  # --- 4. Crop vector layers to window ---
  lat_grid_crop <- sf::st_intersection(lat_grid, window_sf)
  lat_t4grid_nophaseV_crop <- sf::st_intersection(lat_t4grid_nophaseV, window_sf)
  lat_t4grid_phaseV_crop <- sf::st_intersection(lat_t4grid_phaseV, window_sf)
  lat_outline_crop <- sf::st_intersection(lat_outline, window_sf)
  lat_cave_crop <- sf::st_intersection(lat_cave, window_sf)
  lat_dripline_crop <- sf::st_intersection(lat_dripline, window_sf)
  lat_limestone_crop <- sf::st_intersection(lat_limestone, window_sf)
  
  # --- 5. Prepare square labels ---
  label_squares <- lat_t4grid_nophaseV_crop %>%
    filter(SQ_ID %in% c("J2","J3","J4","J5","J6","K6","L6","M6","N6")) %>%
    st_centroid() %>%
    mutate(
      label_text = case_when(
        SQ_ID %in% c("J2","J3","J4","J5") ~ substr(SQ_ID, 2, 2),
        SQ_ID == "J6" ~ "J",
        SQ_ID %in% c("K6","L6","M6","N6") ~ substr(SQ_ID, 1, 1)
      ),
      nudge = case_when(
        SQ_ID %in% c("J2","J3","J4","J5") ~ -0.5,
        SQ_ID == "J6" ~ 0.5,
        SQ_ID %in% c("K6","L6","M6","N6") ~ 0.5
      )
    ) %>%
    bind_rows(
      lat_t4grid_nophaseV_crop %>%
        filter(SQ_ID == "J6") %>%
        st_centroid() %>%
        mutate(
          label_text = "6",
          nudge = -0.5
        )
    )
  # --- 6. Define the title - Format type_name for the title  ---
  plot_name <- dplyr::recode(
    type_name,
    "mammals" = "Mammals",
    "aves" = "Aves",
    "burnt" = "Burnt",
    "unburnt" = "Unburnt",
    "nocrust" = "No Crusts",
    "crust" = "Crusts",
    .default = type_name  # fallback
  )
  
  plot_title <- paste("Latnija Phase V Fauna:", plot_name)
  
  # --- 7. Build the map ---
  map <- ggplot() +
    # Patterned limestone
    ggpattern::geom_sf_pattern(
      data = lat_limestone_crop,
      aes(geometry = geometry),
      inherit.aes = FALSE,
      fill = NA,
      color = "black",
      pattern = "image",
      pattern_filename = pattern_file,
      pattern_type = "tile",
      pattern_scale = 1.5,
      pattern_aspect_ratio = 1
    ) +
    
    # Points (all of Phase V) - Add if required
    geom_sf(data = pts_type, color = "purple", size = 0.05, alpha = 1,show.legend = FALSE) +
    
    
    # Grids and outlines
    geom_sf(data = lat_grid_crop, fill = "grey", alpha = 0.6, color = "grey60", linewidth = 0.3) +
    geom_sf(data = lat_cave_crop, fill = NA, color = "black", linewidth = 0.8) +
    geom_sf(data = lat_t4grid_phaseV_crop, fill = NA, color = "grey60", linewidth = 0.3) +
    geom_sf(data = lat_t4grid_nophaseV_crop, fill = "lightgrey", alpha = 0.3, color = NA, linewidth = 0.3) +
    # Patterned rocks with legend
    ggpattern::geom_sf_pattern(
      data = lat_t4rocks,
      aes(geometry = geometry, pattern = "Rock"),
      inherit.aes = FALSE,
      fill = NA,
      color = "black",
      linewidth = 0.5,
      pattern_density = 0.03,
      pattern_spacing = 0.02,
      show.legend = TRUE
    ) +
    scale_pattern_manual(
      name = "",
      values = c("Rock" = "stripe"),
      guide = guide_legend(order = 1)
    ) +
    geom_sf(data = lat_t4grid_nophaseV_crop, fill = NA, alpha = 0.3, color = "grey60", linewidth = 0.3) +
    geom_sf(data = lat_dripline_crop, fill = NA, color = "black", linewidth = 0.6, linetype = "dashed") +
    geom_sf(data = lat_outline_crop, fill = NA, color = "black", linewidth = 0.6) +
    # Square labels
    geom_sf_text(
      data = label_squares,
      aes(label = label_text),
      nudge_x = label_squares$nudge,
      nudge_y = -0.6,
      size = 4,
      fontface = "bold",
      color = "black"
    ) +
    # North arrow
    annotation_north_arrow(
      location = "bl",
      which_north = "true",
      style = north_arrow_orienteering,
      pad_y = unit(0.9, "cm"),
      width = unit(0.5, "cm"),
      height = unit(1, "cm")
    ) +
    annotation_scale(
      location = "bl",
      width_hint = 0.2,
      text_cex = 1,
      text_face = "bold",
    ) +
    coord_sf(
      datum = sf::st_crs(lat_outline),
      expand = FALSE,
      clip = "on"
    ) +
    ggtitle(plot_title) +
    theme_minimal() +
    theme(
      plot.title = element_text(face = "bold", hjust = 0.5, size = 14),
      panel.grid = element_blank(),
      axis.ticks.length = unit(0.1, "cm"),
      axis.ticks = element_line(color = "black", linewidth = 0.5),
      axis.text = element_text(size = 8, color = "black"),
      axis.title = element_blank(),
      legend.position = "right",
      panel.border = element_rect(colour = "black", fill=NA, linewidth=1)
    )
  
  return(map)
}


# run function and save plots to folder 'Planview_Maps'
for (type_name in names(kde_all)) {
  # Create output directory
  out_dir <- "outputs/Planview_Maps"
  if (!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)
  
  p <- plot_planview_type(
    points_sf = lat,
    type_name = type_name,
    lat_grid = lat_grid,
    lat_t4grid_nophaseV = lat_t4grid_nophaseV,
    lat_t4grid_phaseV = lat_t4grid_phaseV,
    lat_outline = lat_t4outline,
    lat_window = lat_window,
    lat_cave = lat_cave,
    lat_limestone = lat_limestone,
    lat_t4rocks = lat_t4rocks,
    lat_dripline = lat_dripline
  )
  
  # Plot
  print(p)
  
  # Save to file
  ggsave(
    filename = file.path(out_dir, paste0("Planview_Map_", type_name, ".png")),
    plot = p,
    width = 6,
    height = 6,
    dpi = 600
  )
  cat("Rendering output for:", type_name, "to: ", normalizePath(out_dir), "\n")
}

# Run a single plot in R to view
plot_planview_type(
  points_sf = lat,
  type_name = 'nocrust', # Change this to desired 'mark' (dont forget the '')
  lat_grid = lat_grid,
  lat_t4grid_nophaseV = lat_t4grid_nophaseV,
  lat_t4grid_phaseV = lat_t4grid_phaseV,
  lat_outline = lat_t4outline,
  lat_window = lat_window,
  lat_cave = lat_cave,
  lat_limestone = lat_limestone,
  lat_t4rocks = lat_t4rocks,
  lat_dripline = lat_dripline
)



### ----------------------------------------------------------------------------
## All Faunal finds, including specimens <20mm from sieves with plotted finds
# Since no piece provenience data are available for sieved finds, these data are 
# summarised as counts within each grid square.
### ----------------------------------------------------------------------------

# Join Lat faunal counts with the lat_t4grid shapefile

lat_map <- lat_t4grid %>%
  left_join(lat_fauna_sq, by = "SQ_ID") # join faunal data (by square) with shapefile of excavation squares
lat_map # inspect


# 1) Function to plot counts per square

plot_count_type <- function(
    lat_map,                  
    points_sf,
    type_name,
    lat_grid,
    lat_outline,
    lat_window,
    lat_cave,
    lat_limestone,
    pattern_file = "textures/limestone.png"
) {
  
  # --- 1. Subset points by type ---
  pts_type <- points_sf[points_sf$type == type_name, ]
  
  # --- 2. Convert window + outline ---
  window_poly <- as.polygonal(lat_window)
  window_sf <- sf::st_as_sf(window_poly)
  sf::st_crs(window_sf) <- sf::st_crs(points_sf)
  
  outline_sf <- sf::st_as_sf(lat_outline)
  sf::st_crs(outline_sf) <- sf::st_crs(points_sf)
  
  # --- 3. Crop all spatial layers ---
  lat_grid_crop            <- sf::st_intersection(lat_grid, window_sf)
  lat_outline_crop         <- sf::st_intersection(lat_outline, window_sf)
  lat_cave_crop            <- sf::st_intersection(lat_cave, window_sf)
  lat_limestone_crop       <- sf::st_intersection(lat_limestone, window_sf)
  lat_map_crop             <- sf::st_intersection(lat_map, window_sf)
  
  # --- 4. Square labels (your exact logic) ---
  label_squares <- lat_map_crop %>%
    filter(SQ_ID %in% c("J2","J3","J4","J5","J6","K6","L6","M6","N6")) %>%
    st_centroid() %>%
    mutate(
      label_text = case_when(
        SQ_ID %in% c("J2","J3","J4","J5") ~ substr(SQ_ID, 2, 2),
        SQ_ID == "J6"                    ~ "J",
        SQ_ID %in% c("K6","L6","M6","N6") ~ substr(SQ_ID, 1, 1)
      ),
      nudge = case_when(
        SQ_ID %in% c("J2","J3","J4","J5") ~ -0.5,
        SQ_ID == "J6"                    ~ 0.5,
        SQ_ID %in% c("K6","L6","M6","N6") ~ 0.5
      )
    ) %>%
    bind_rows(
      lat_map_crop %>%
        filter(SQ_ID == "J6") %>%
        st_centroid() %>%
        mutate(label_text = "6", nudge = -0.5)
    )
  
  # --- 5. Title formatting ---
  plot_name <- dplyr::recode(
    type_name,
    "mammals"      = "Mammals",
    "aves"         = "Aves",
    "burnt"        = "Burnt",
    "unburnt"      = "Unburnt",
    "nocrust" = "No Crusts",
    "crust"   = "Crusts",
    .default = type_name
  )
  
  plot_title <- paste("Latnija Phase V Fauna:", plot_name)
  
  # --- 6. BUILD FINAL MAP ---
  map <- ggplot() +
    
    # ----- LIMESTONE -----
  ggpattern::geom_sf_pattern(
    data = lat_limestone_crop,
    aes(geometry = geometry),
    inherit.aes = FALSE,
    fill = NA, color = "black",
    pattern = "image",
    pattern_filename = pattern_file,
    pattern_type = "tile",
    pattern_scale = 1.5
  ) +
    
    # ----- FAUNAL CHOROPLETH (FROM lat_map) -----
  geom_sf(
    data = lat_map_crop,
    aes(fill = .data[[type_name]]),
    color = "black",
    linewidth = 0.2
  ) +
    
    # ----- NUMERIC COUNT LABELS -----
  geom_sf_text(
    data = lat_map_crop %>% filter(.data[[type_name]] > 0),
    aes(label = .data[[type_name]]),
    size = 3,
    fontface = "bold",
    color = "white"
  ) +
    
    # ----- OUTLINES -----
  geom_sf(data = lat_cave_crop, fill = NA, color = "black", linewidth = 0.8) +
    geom_sf(data = lat_outline_crop, fill = NA, color = "black", linewidth = 0.6) +
    
    # ----- GRID LABELS -----
  geom_sf_text(
    data = label_squares,
    aes(label = label_text),
    nudge_x = label_squares$nudge,
    nudge_y = -0.6,
    size = 4,
    fontface = "bold"
  ) +
    
    # ----- SCALE & NORTH ARROW -----
  annotation_north_arrow(
    location = "bl",
    which_north = "true",
    style = north_arrow_orienteering,
    pad_y = unit(0.9, "cm"),
    width = unit(0.5, "cm"),
    height = unit(1, "cm")
  ) +
    annotation_scale(location = "bl", width_hint = 0.2) +
    
    coord_sf(datum = sf::st_crs(lat_outline), expand = FALSE) +
    
    scale_fill_viridis_c(option = "D", na.value = "white") +
    
    labs(
      title = plot_title,
      fill  = plot_name
    ) +
    
    theme_minimal() +
    theme(
      plot.title = element_text(face = "bold", hjust = 0.5, size = 14),
      panel.grid = element_blank(),
      axis.ticks.length = unit(0.1, "cm"),
      axis.ticks = element_line(color = "black", linewidth = 0.5),
      axis.text = element_text(size = 8, color = "black"),
      axis.title = element_blank(),
      legend.position = "right",
      panel.border = element_rect(colour = "black", fill=NA, linewidth=1)
    )
  
  return(map)
}


## 2) View all maps in R and save to folder Fauna_counts

count_vars <- c(
  "mammals",
  "aves",
  "burnt",
  "unburnt",
  "nocrust",
  "crust"
)

for (type_name in count_vars) {
  out_dir <- "outputs/Fauna_counts"
  if (!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)
  # Generate the plot
  p <- plot_count_type(
    lat_map = lat_map,
    points_sf = lat,
    type_name = type_name,
    lat_grid = lat_grid,
    lat_outline = lat_t4outline,
    lat_window = lat_window,
    lat_cave = lat_cave,
    lat_limestone = lat_limestone
  )
  
  # Print immediately
  print(p)
  
  # Save the plot
  ggsave(
    filename = file.path(out_dir, paste0("Counts_", type_name, ".png")),
    plot = p,
    width = 6,
    height = 6,
    dpi = 600,
  )
  
  cat("Saved Fauna count plot for:", type_name, "to", normalizePath(out_dir), "\n")
}

# plot single map (change type_name to required type) if needed

plot_count_type(
  lat_map = lat_map,            
  points_sf = lat,
  type_name = "mammals", # Change to required 'type'
  lat_grid = lat_grid,
  lat_outline = lat_t4outline,
  lat_window = lat_window,
  lat_cave = lat_cave,
  lat_limestone = lat_limestone
)

### ----------------------------------------------------------------------------
## Latnija Refits
### ----------------------------------------------------------------------------

## 1) Read CSV and convert to sf object ---
refits <- read.csv("data/lat.phase.v.refits.csv", stringsAsFactors = FALSE)

## 2) Convert to sf (UTM Zone 33N = EPSG:32633) ---
refits_sf <- st_as_sf(refits, coords = c("Longitude", "Latitude"), crs = 32633)

# 3) Compute refit lines ---
# For each REFIT.SET, collect all points and connect them with a line
refit_lines_sf <- refits_sf %>%
  group_by(REFIT.SET) %>%
  summarise(do_union = FALSE) %>%
  st_cast("LINESTRING")

# 4) Compute distances between refit points ---
# Extract coordinates (to include Altitude)
coords <- refits %>% select(REFIT.SET, Longitude, Latitude, Altitude)

Refit_Distance_Summary <- coords %>%
  group_by(REFIT.SET) %>%
  summarise(
    n_points = n(),
    
    # Mean horizontal distance
    horiz_dist_mean = {
      if (n() > 1) {
        d <- dist(as.matrix(pick(Longitude, Latitude)))
        mean(as.numeric(d))
      } else { NA }
    },
    
    # Total horizontal distance (sum of pairwise distances)
    horiz_dist_total = {
      if (n() > 1) {
        d <- dist(as.matrix(pick(Longitude, Latitude)))
        sum(as.numeric(d))
      } else { NA }
    },
    
    # Mean vertical distance
    vert_dist_mean = {
      if (n() > 1) {
        d <- dist(as.matrix(pick(Altitude)))
        mean(as.numeric(d))
      } else { NA }
    },
    
    # Maximum vertical distance
    vert_dist_max = {
      if (n() > 1) {
        d <- dist(as.matrix(pick(Altitude)))
        max(as.numeric(d))
      } else { NA }
    }
  ) %>%
  mutate(across(where(is.numeric), ~ round(.x, 3)))

# 5) Overall averages across all sets ---
overall_refit_summary <- Refit_Distance_Summary %>%
  summarise(
    mean_horizontal = mean(horiz_dist_mean, na.rm = TRUE),
    mean_vertical = mean(vert_dist_mean, na.rm = TRUE),
    max_horizontal = max(horiz_dist_mean, na.rm = TRUE),
    max_vertical = max(vert_dist_mean, na.rm = TRUE)
  )

print(Refit_Distance_Summary)
print(overall_refit_summary)

# Define output folder
refit_dir <- "outputs/Refits"
# Create folder if it doesn't exist
if (!dir.exists(refit_dir)) dir.create(refit_dir, recursive = TRUE)

# Save CSV
write.csv(Refit_Distance_Summary, file.path(refit_dir, "Lat_PhaseV_Refit_Summary.csv"), row.names = FALSE)


## -----------------------------------------------------------------------------
# Function to plot refit sets as lines and points

plot_refit_map <- function(
    refit_lines_sf,
    refits_sf,
    lat_grid,
    lat_t4grid_nophaseV,
    lat_t4grid_phaseV,
    lat_outline,
    lat_window,
    lat_cave,
    lat_limestone,
    lat_t4rocks,
    overall_refit_summary,
    pattern_file = "textures/limestone.png"
) {
  
  # --- Crop site layers to the refit area ---
  window_poly <- as.polygonal(lat_window)
  window_sf <- st_as_sf(window_poly)
  st_crs(window_sf) <- st_crs(refits_sf)
  
  lat_grid_crop <- st_intersection(lat_grid, window_sf)
  lat_t4grid_nophaseV_crop <- st_intersection(lat_t4grid_nophaseV, window_sf)
  lat_t4grid_phaseV_crop <- st_intersection(lat_t4grid_phaseV, window_sf)
  lat_outline_crop <- st_intersection(lat_outline, window_sf)
  lat_cave_crop <- st_intersection(lat_cave, window_sf)
  lat_limestone_crop <- st_intersection(lat_limestone, window_sf)
  
  # --- Prepare annotation text with title ---
  annotation_text <- paste0(
    "Phase V Refits: \n",
    "Mean hz distance: ", round(overall_refit_summary$mean_horizontal, 2), " m\n",
    "Max hz distance: ", round(overall_refit_summary$max_horizontal, 2), " m\n",
    "Mean z distance: ", round(overall_refit_summary$mean_vertical, 2), " m\n",
    "Max z distance: ", round(overall_refit_summary$max_vertical, 2), " m"
  )
  
  # --- Prepare square labels ---
  label_squares <- lat_t4grid_nophaseV_crop %>%
    filter(SQ_ID %in% c("J2","J3","J4","J5","J6","K6","L6","M6","N6")) %>%
    st_centroid() %>%
    mutate(
      label_text = case_when(
        SQ_ID %in% c("J2","J3","J4","J5") ~ substr(SQ_ID, 2, 2),  # just numbers
        SQ_ID == "J6" ~ "J",                                       # letter only
        SQ_ID %in% c("K6","L6","M6","N6") ~ substr(SQ_ID, 1, 1)    # just letters
      ),
      nudge = case_when(
        SQ_ID %in% c("J2","J3","J4","J5") ~ -0.5,  # numbers: left
        SQ_ID == "J6" ~ 0.5,                       # J on right
        SQ_ID %in% c("K6","L6","M6","N6") ~ 0.5   # letters: right
      )
    ) %>%
    # Add extra row for the "6" next to J6
    bind_rows(
      lat_t4grid_nophaseV_crop %>%
        filter(SQ_ID == "J6") %>%
        st_centroid() %>%
        mutate(
          label_text = "6",   # number label
          nudge = -0.5        # place to the left
        )
    )
  
  # --- Create map ---
  map <- ggplot() +
    # Limestone background
    # --- Limestone pattern (image) ---
    geom_sf_pattern(
      data = lat_limestone_crop,
      aes(geometry = geometry),
      inherit.aes = FALSE,
      fill = NA,
      color = "black",
      pattern = "image",
      pattern_filename = pattern_file,
      pattern_type = "tile",
      pattern_scale = 1.5,
      pattern_aspect_ratio = 1
    ) +
    # Patterned rock polygon with legend
    ggpattern::geom_sf_pattern(
      data = lat_t4rocks,
      aes(geometry = geometry, pattern = factor("Rock")),
      inherit.aes = FALSE,
      fill = NA,
      color = "black",
      linewidth = 0.5,
      pattern_density = 0.03,
      pattern_spacing = 0.02,
      show.legend = TRUE
    ) +
    scale_pattern_manual(
      name = "",
      values = c("Rock" = "stripe")
    ) +
    # Site context
    geom_sf(data = lat_grid_crop, fill = "grey", alpha = 0.6, color = "grey60", linewidth = 0.3,show.legend = FALSE) +
    geom_sf(data = lat_t4grid_nophaseV_crop, fill = "lightgrey", alpha = 0.3, color = "grey60", linewidth = 0.3,show.legend = FALSE) +
    geom_sf(data = lat_t4grid_phaseV_crop, fill = NA, color = "grey60", linewidth = 0.3,show.legend = FALSE) +
    geom_sf(data = lat_outline_crop, fill = NA, color = "black", linewidth = 0.6,show.legend = FALSE) +
    geom_sf(data = lat_cave_crop, fill = NA, color = "black", linewidth = 0.8,show.legend = FALSE) +
    
    # Refit lines + points
    geom_sf(data = lat_phasev, color = "black", size = 0.05, alpha = 1,show.legend = FALSE) +
    geom_sf(data = refits_sf, color = "orange", size = 1, fill = "orange",shape = 23, alpha = 1,show.legend = FALSE) +
    geom_sf(data = refit_lines_sf, color = "red", linewidth = 0.6,show.legend = FALSE) +
    
    # Square labels
    geom_sf_text(
      data = label_squares,
      aes(label = label_text),
      nudge_x = label_squares$nudge,  
      nudge_y = -0.5,                 
      size = 4,
      fontface = "bold",
      color = "black",
      show.legend = FALSE
    ) +
    
    # Remove color legend for REFIT.SET
    guides(color = "none") +
    
    # Add semi-transparent summary annotation
    geom_label(
      data = data.frame(x = Inf, y = Inf, label = annotation_text),
      aes(x = x, y = y, label = label),
      hjust = 1.1,       
      vjust = 1.4,         
      size = 4,
      fill = "white",
      alpha = 0.7,
      label.size = 0.3,
      show.legend = FALSE
    ) +
    annotation_north_arrow(
      location = "bl",         # top-right corner
      which_north = "true",    # true north
      style = north_arrow_orienteering,
      pad_y = unit(0.9, "cm"),
      width = unit(1, "cm"),
      height = unit(1.5, "cm")
    ) +
    annotation_scale(
      location = "bl",
      width_hint = 0.2,
      text_cex = 1,
      text_face = "bold",
    )+
    labs(title = "Latnija Phase V Fauna: Refits") +
    coord_sf(datum = st_crs(lat_outline), expand = FALSE, clip = "on") +
    theme_minimal() +
    theme(
      plot.title = element_text(hjust = 0.5, size = 12, face = "bold"),
      panel.grid = element_blank(),
      axis.ticks.length = unit(0.1, "cm"),
      axis.ticks = element_line(color = "black", linewidth = 0.5),
      axis.text = element_text(size = 8, color = "black"),
      axis.title = element_blank(),
      legend.position = "right",
      panel.border = element_rect(colour = "black", fill=NA, linewidth=1)
    )
  
  return(map)
}

# Create and assign the map object
map_plot <- plot_refit_map(
  refit_lines_sf = refit_lines_sf,
  refits_sf = refits_sf,
  lat_grid = lat_grid,
  lat_t4grid_nophaseV = lat_t4grid_nophaseV,
  lat_t4grid_phaseV = lat_t4grid_phaseV,
  lat_outline = lat_t4outline,
  lat_window = lat_window,
  lat_cave = lat_cave,
  lat_limestone = lat_limestone,
  lat_t4rocks = lat_t4rocks,
  overall_refit_summary = overall_refit_summary
)
map_plot # View the map

# Save to TIF
ggsave("outputs/Refits/PhaseV_Refits_Map.png", plot = map_plot, device = "png", dpi = 300, width = 8, height = 8)


### ----------------------------------------------------------------------------
## Kscaled and Lscaled functions,
### ----------------------------------------------------------------------------

run_inhom_KL_tests <- function(
    sf_points,
    type_field = "type",
    out_dir = "outputs/inhom_KL_envelopes",
    nsim = 99,
    min_n = 5,
    min_p = 0.001,
    rmax = NULL
) {
  # Create output folder
  if(!dir.exists(out_dir)) dir.create(out_dir)
  
  types <- unique(sf_points[[type_field]])
  
  # Summary table
  summary_df <- data.frame(
    type = character(),
    n_points = numeric(),
    DCLF_K = numeric(),
    MAD_K = numeric(),
    DCLF_L = numeric(),
    MAD_L = numeric(),
    stringsAsFactors = FALSE
  )
  
  # convert sf to ppp
  sf_to_ppp <- function(sub_sf) {
    coords <- sf::st_coordinates(sub_sf)
    bb <- sf::st_bbox(sub_sf)
    win <- spatstat.geom::owin(
      xrange = c(bb["xmin"], bb["xmax"]),
      yrange = c(bb["ymin"], bb["ymax"])
    )
    ppp_obj <- spatstat.geom::ppp(x = coords[,1], y = coords[,2], window = win)
    return(ppp_obj)
  }
  
  # Loop through types
  for(t in types) {
    cat("Processing:", t, "\n")
    
    sub_sf <- sf_points[sf_points[[type_field]] == t, ]
    
    if(nrow(sub_sf) < min_n) {
      cat("Skipping", t, "- too few points\n")
      next
    }
    
    ppp_obj <- sf_to_ppp(sub_sf)
    
    # Lscaled envelope
    env_L <- spatstat.explore::envelope(
      ppp_obj, Lscaled, nsim = nsim, fix.n = TRUE,
      correction = "best", rank = 1, global = TRUE
    )
    png(file.path(out_dir, paste0("Lscaled_", t, ".png")), width = 800, height = 600)
    plot(env_L, lwd = 3, main = paste("Lscaled:", t))
    dev.off()
    
    # Kinhom envelope
    env_K <- spatstat.explore::envelope(
      ppp_obj, Kinhom, nsim = nsim, fix.n = TRUE,
      correction = "border", global = TRUE, lambda = NULL
    )
    png(file.path(out_dir, paste0("Kinhom_", t, ".png")), width = 800, height = 600)
    plot(env_K, lwd = 3, main = paste("Kinhom:", t))
    dev.off()
    
    # DCLF and MAD tests
    dclf_K <- dclf.test(ppp_obj, Kinhom, nsim = nsim, rmax = rmax, use.theo = TRUE)
    mad_K <- mad.test(ppp_obj, Kinhom, nsim = nsim, rmax = rmax, use.theo = TRUE)
    dclf_L <- dclf.test(ppp_obj, Lscaled, nsim = nsim, rmax = rmax, use.theo = TRUE)
    mad_L <- mad.test(ppp_obj, Lscaled, nsim = nsim, rmax = rmax, use.theo = TRUE)
    
    # Add to summary
    summary_df <- rbind(
      summary_df,
      data.frame(
        type = t,
        n_points = nrow(sub_sf),
        DCLF_K = max(dclf_K$p.value, min_p),
        MAD_K  = max(mad_K$p.value, min_p),
        DCLF_L = max(dclf_L$p.value, min_p),
        MAD_L  = max(mad_L$p.value, min_p)
      )
    )
  }
  
  return(summary_df)
}

# Run function
inhom_test_summary <- run_inhom_KL_tests(
  sf_points = lat,
  type_field = "type",
  out_dir = "outputs/inhom_KL_envelopes",
  nsim = 99,
  min_n = 15,
  min_p = 0.001
)

print(inhom_test_summary)

# Add a summary of strength of results
get_significance <- function(p) {
  if(p <= 0.001) return("***")      # strong
  else if(p <= 0.01) return("**")   # moderate
  else if(p <= 0.05) return("*")    # weak
  else return("ns")                  # not significant
}

# Apply to all test columns
inhom_test_summary$DCLF_K_sig <- sapply(inhom_test_summary$DCLF_K, get_significance)
inhom_test_summary$MAD_K_sig  <- sapply(inhom_test_summary$MAD_K, get_significance)
inhom_test_summary$DCLF_L_sig <- sapply(inhom_test_summary$DCLF_L, get_significance)
inhom_test_summary$MAD_L_sig  <- sapply(inhom_test_summary$MAD_L, get_significance)

# Reorder columns: numeric result followed by its significance
inhom_test_summary <- inhom_test_summary[, c(
  "type",
  "n_points",
  "DCLF_K", "DCLF_K_sig",
  "MAD_K",  "MAD_K_sig",
  "DCLF_L", "DCLF_L_sig",
  "MAD_L",  "MAD_L_sig"
)]

# Inspect the updated table
print(inhom_test_summary)

# Save csv
write.csv(inhom_test_summary, "outputs/inhom_KL_envelopes/inhom_KL_test_summary.csv", row.names = FALSE)


### ----------------------------------------------------------------------------
## Pairwise inhomogeneous cross-type spatial interaction tests

# Define function
run_inhom_cross <- function(
    sf_points,
    type_field = "type",
    mark_pairs = list(
      c("mammals", "aves"),
      c("crust", "nocrust"),
      c("burnt", "unburnt")
    ),
    out_dir = "outputs/inhom_cross_envelopes",
    nsim = 99,
    min_n = 5,
    rmax = NULL   # set distance for envelopes
) {
  if(!dir.exists(out_dir)) dir.create(out_dir)
  
  # convert sf to ppp
  sf_to_ppp <- function(sub_sf) {
    coords <- sf::st_coordinates(sub_sf)
    bb <- sf::st_bbox(sub_sf)
    win <- spatstat.geom::owin(
      xrange = c(bb["xmin"], bb["xmax"]),
      yrange = c(bb["ymin"], bb["ymax"])
    )
    marks <- factor(sub_sf[[type_field]])
    ppp_obj <- spatstat.geom::ppp(
      x = coords[,1], y = coords[,2], window = win, marks = marks
    )
    return(ppp_obj)
  }
  
  summary_df <- data.frame(
    pair = character(),
    n_points_1 = numeric(),
    n_points_2 = numeric(),
    stringsAsFactors = FALSE
  )
  
  for(pair in mark_pairs) {
    cat("Processing pair:", pair[1], "vs", pair[2], "\n")
    
    sub_sf <- sf_points[sf_points[[type_field]] %in% pair, ]
    
    if(nrow(sub_sf) < min_n) {
      cat("Skipping pair - too few points\n")
      next
    }
    
    ppp_obj <- sf_to_ppp(sub_sf)
    
    # Ensure both marks are present
    if(length(levels(ppp_obj$marks)) < 2) {
      cat("Skipping pair - one of the marks missing\n")
      next
    }
    
    # Cross-type Kinhom
    png(file.path(out_dir, paste0("Kcross_inhom_", pair[1], "_vs_", pair[2], ".png")),
        width = 800, height = 600)
    env_Kcross <- spatstat.explore::envelope(
      ppp_obj,
      fun = function(X, r = NULL, ...) {
        rvals <- if(!is.null(r)) r else NULL
        Kcross.inhom(X, i = pair[1], j = pair[2], r = rvals, ...)
      },
      nsim = nsim,
      global = TRUE,
      correction = "best",
      r = if(!is.null(rmax)) seq(0, rmax, length.out = 100) else NULL
    )
    plot(env_Kcross, lwd = 3, main = paste("Inhom. Cross K:", pair[1], "vs", pair[2]))
    dev.off()
    
    # Cross-type Linhom
    png(file.path(out_dir, paste0("Lcross_inhom_", pair[1], "_vs_", pair[2], ".png")),
        width = 800, height = 600)
    env_Lcross <- spatstat.explore::envelope(
      ppp_obj,
      fun = function(X, r = NULL, ...) {
        rvals <- if(!is.null(r)) r else NULL
        Lcross.inhom(X, i = pair[1], j = pair[2], r = rvals, ...)
      },
      nsim = nsim,
      global = TRUE,
      correction = "best",
      r = if(!is.null(rmax)) seq(0, rmax, length.out = 100) else NULL
    )
    plot(env_Lcross, lwd = 3, main = paste("Inhom. Cross L:", pair[1], "vs", pair[2]))
    dev.off()
    
    # Summary table
    summary_df <- rbind(
      summary_df,
      data.frame(
        pair = paste(pair, collapse = " vs "),
        n_points_1 = sum(sub_sf[[type_field]] == pair[1]),
        n_points_2 = sum(sub_sf[[type_field]] == pair[2])
      )
    )
  }
  
  return(summary_df)
}

# Run function
cross_summary <- run_inhom_cross(
  sf_points = lat,
  type_field = "type",
  out_dir = "outputs/inhom_cross_envelopes",
  nsim = 99,
  min_n = 15,
  rmax = 2   # only compute distances up to 2 m
)
print(cross_summary)


### ----------------------------------------------------------------------------
##  X2 tests:
## Quadrat grids of varying resolution (5×5, 8×8, 10×10, and 12×12) are applied 
## to examine the effect of grid size on the detection of clustering or dispersion.
### ----------------------------------------------------------------------------

# 1) define ppp object
pp <- select(lat, long, lat, type) # select required fields from lat shapefile
pp$type <- as.factor(pp$type) # set 'type' to factor

ppp.lat <- ppp(pp$long, pp$lat, lat_window, marks = pp$type)

any(duplicated(ppp.lat))
ppp.lat.split <- unique(ppp.lat)
unitname(ppp.lat.split) <- "m"

# 2) Function to run quadrat tests across multiple types and grid sizes
run_quadrat_tests <- function(ppp_obj, grids = c(5, 8, 10, 12)) {
  # Ensure marks are factors
  if (!is.factor(ppp_obj$marks)) {
    ppp_obj$marks <- as.factor(ppp_obj$marks)
  }
  
  # Split the ppp object by type
  ppp_split <- split(ppp_obj)
  
  results <- data.frame()
  
  for (grid in grids) {
    for (type_name in names(ppp_split)) {
      # Run quadrat.test safely (suppress warnings)
      qt <- suppressWarnings(quadrat.test(ppp_split[[type_name]], nx = grid, ny = grid))
      
      # Extract results
      temp <- data.frame(
        type = type_name,
        grid_size = grid,
        X2 = qt$statistic,
        df = qt$parameter,
        p_value = ifelse(qt$p.value < 0.001, "<0.001", qt$p.value),
        significant = qt$p.value < 0.05
      )
      
      results <- rbind(results, temp)
    }
  }
  
  # Sort by grid size then type
  results <- results[order(results$grid_size, results$type), ]
  
  # Reset row names
  rownames(results) <- NULL
  
  return(results)
}


# 3) run the function
quadrat_summary <- run_quadrat_tests(ppp.lat.split)

print(quadrat_summary) #view results

quadrat_summary_grouped <- quadrat_summary %>%
  group_by(type) %>%        # Group by type
  arrange(grid_size, .by_group = TRUE)  # Arrange by grid size within each type

# 4) View results and save
print(quadrat_summary_grouped)
# Create output folder
x2_dir <- "outputs/x2_tests"
if(!dir.exists(x2_dir)) dir.create(x2_dir)
write.csv(quadrat_summary_grouped, "outputs/x2_tests/quadrat_summary.csv", row.names = FALSE)

